' EXAMPLE:
'   This example demonstrates how you can implement your own OPC Server
'   Interfaces within your own new/existing applications.
'
'   There are 6 OPC Tags exposed:
'       3, representing the "user" controls on the left of the form.
'       3, representing the "simulated" controls on the right of the form.
'
'   A Timer will generate random values that are then fed into the SIMULATED
'   form controls. These form controls will raise events when their state/value
'   changes. Event Handlers are attached each control so that an update can
'   be made to the applicable OPC Item (tag).
'   
'   An OPC Server, upon receiving an updated OPC Item value, will automatically
'   send a notification event to any OPC Client that has "subscribed" to the
'   OPC Items.
'
'   The "user" controls are also setup with event handlers to achieve the same
'   behavior as described above.
'
' VENDOR:
'   Software Toolbox, Inc. http://www.softwaretoolbox.com/ 
'
' TOOLKIT INFORMATION:
'   SLIK-DA OPC Server Toolkit
'   http://www.nordyn.com/ 
'
' DISCLAIMERS:
'   This example is provided as-is and is not intended for production-use,
'   and is purely for educational purposes only to demonstrate the concept.
'
Imports System.ComponentModel
Imports NDI.SLIKDA.Interop
Public Class VBDotNET_Simple_Server

    ' we will use a random number generator
    Private randomizer As Random = New Random()

    ' this variable will hold our tag-database
    Private myOpcTags As ISLIKTags


#Region "   Other, not relevant code"

    Private Sub LinkLabel1_LinkClicked(ByVal sender As System.Object, ByVal e As System.Windows.Forms.LinkLabelLinkClickedEventArgs) Handles LinkLabel1.LinkClicked
        Try
            System.Diagnostics.Process.Start("http://www.softwaretoolbox.com/quicklink.asp?partnumber=41262110")
        Catch
        End Try
    End Sub

#End Region

#Region "   FORM Control Events "

    'This code is executed when the form loads
    Private Sub Form1_Load(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles MyBase.Load

        Dim bExit As Boolean
        bExit = False

        ' Get the command line arguments to check for server registration switches
        Dim sCmdLine As String
        sCmdLine = Microsoft.VisualBasic.Command()

        '
        ' NOTE: Do NOT prefix server registration switches with a forward slash "/".
        ' The VB runtime engine with NOT pass such arguments to the application.
        '
        'Call LogDebugMessage("Command line arguments: '" & sCmdLine & "'")

        '
        ' Perform server registration as required.
        '
        If InStr(1, sCmdLine, "unregserver", CompareMethod.Text) > 0 Then
            SlikServer1.UnregisterServer()
            bExit = True
        ElseIf InStr(1, sCmdLine, "regserver", CompareMethod.Text) > 0 Then
            SlikServer1.RegisterServer()
            bExit = True
        Else
            ' ignore any other switches
        End If

        If bExit Then
            ' If a server registration switch was specified in the command line, the
            ' standard COM server behaviour is to exit immediately after changing the
            ' registration information.

            'Me.Close()
            'Exit Sub
        Else
            ' create a new instance of an empty tag database
            myOpcTags = SlikServer1.SLIKTags

            ' define a simple read/write access-right, saves us OR'ing this
            ' enumeration multiple times
            Dim readWriteAccess As AccessPermissionsEnum = NDI.SLIKDA.Interop.AccessPermissionsEnum.sdaReadAccess Or AccessPermissionsEnum.sdaWriteAccess

            ' A Tag name is "just" a name. But when you prefix that name with a '.'
            ' then the OPC Server toolkit will automatically place that tag within
            ' a group. This simplifies the OPC Client experience when they browse
            ' your OPC Servers address-space.
            ' In this case, we will defined two groups:
            '  (a) USER
            '  (b) SIMULATION
            ' .... you will see these are prefixes to the tagnames.


            ' add the user-tags. These are the tags that represent the 3 form
            ' controls on the LEFT side of the form
            myOpcTags.Add("User.CheckBox", readWriteAccess, 0, 0, DateTime.Now)
            myOpcTags.Add("User.Numeric", readWriteAccess, 0, 0, DateTime.Now)
            myOpcTags.Add("User.TextBox", readWriteAccess, 0, 0, DateTime.Now)

            ' add the simulation tags. These are the tags that represent the 3
            ' form controls on the RIGHT side of the form
            myOpcTags.Add("Simulation.ProgressBar", readWriteAccess, 0, 0, DateTime.Now)
            myOpcTags.Add("Simulation.Numeric", readWriteAccess, 0, 0, DateTime.Now)
            myOpcTags.Add("Simulation.CheckBox", readWriteAccess, 0, 0, DateTime.Now)

            ' setup the tag DATA TYPES of these tags. We could've done these above,
            ' but we've separated them here to illustrate the 2 tag groupings above.
            myOpcTags("User.CheckBox").DataType = VariantType.Boolean
            myOpcTags("User.Numeric").DataType = VariantType.Integer
            myOpcTags("User.TextBox").DataType = VariantType.String
            myOpcTags("Simulation.ProgressBar").DataType = VariantType.Integer
            myOpcTags("Simulation.Numeric").DataType = VariantType.Integer
            myOpcTags("Simulation.CheckBox").DataType = VariantType.Boolean

            ' In this example, we will also register and start the server
            ' Under normal circumstances you would NOT do this, because this is a
            ' step that should be done once and once-only, at the end of an
            ' installation routine. Most OPC Server implementations allow the use
            ' of a command-line argument to call this function.
            'SlikServer1.RegisterServer()

            ' Now START the OPC Server interface. This is called whenever your
            ' application starts.
            SlikServer1.StartServer()

        End If



    End Sub

    ' This code is executed when the EXIT button is clicked.
    Private Sub ExitButton_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles ExitButton.Click
        If (MessageBox.Show("Close the OPC Server and Exit?", "Exit?", MessageBoxButtons.YesNo, MessageBoxIcon.Question, MessageBoxDefaultButton.Button2) = Windows.Forms.DialogResult.Yes) Then
            Try
                ' Notify the OPC Clients that you want to shut down. This gives
                ' each OPC Client enough time to gracefully disconnect.
                SlikServer1.RequestDisconnect("User Requested Shutdown")

                ' This function, just like the ".RegisterServer" as seen in the 
                ' Form_Load() event is a function that should not be called under
                ' normal circumstances. This should be done by the installer
                ' when the application is being removed from the computer.
                'SlikServer1.UnregisterServer()
            Catch ex As Exception
                Debug.WriteLine(ex.Message)
            Finally
                Application.Exit()
            End Try
        End If
    End Sub

    ' When the (right-side) CHECKBOX changes, Start the Timer.
    Private Sub SimulationEnabled_CheckStateChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles SimulationEnabled.CheckStateChanged
        Timer1.Enabled = SimulationEnabled.Checked
    End Sub

    ' Timer interval changer.
    Private Sub NumericUpDown1_ValueChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles NumericUpDown1.ValueChanged
        Timer1.Interval = Convert.ToInt32(NumericUpDown1.Value)
    End Sub

    ' When the Simulated NUMERIC changes, update the OPC Tag.
    Private Sub SimulationNumeric_ValueChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles SimulationNumeric.ValueChanged
        If (Not myOpcTags Is Nothing) Then
            Dim tag As ISLIKTag = myOpcTags("Simulation.Numeric")
            tag.SetVQT(Convert.ToInt32(SimulationNumeric.Value), 192, DateTime.Now)
        End If
    End Sub

    ' When the simulated Checbox changes, update the OPC Tag.
    Private Sub SimulationCheckbox_CheckStateChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles SimulationCheckbox.CheckStateChanged
        If (Not myOpcTags Is Nothing) Then
            myOpcTags("Simulation.CheckBox").SetVQT(SimulationCheckbox.Checked, 192, DateTime.Now)
        End If
    End Sub

    ' When the USER checkbox changes, update the applicable OPC Tag
    Private Sub UserCheckBox_CheckStateChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles UserCheckBox.CheckStateChanged
        If (Not myOpcTags Is Nothing) Then
            myOpcTags("User.CheckBox").SetVQT(UserCheckBox.Checked, 192, DateTime.Now)
        End If
    End Sub

    'When the USER Numeric control changes, update the applicable OPC Tag
    Private Sub UserNumeric_ValueChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles UserNumeric.ValueChanged
        If (Not myOpcTags Is Nothing) Then
            myOpcTags("User.Numeric").SetVQT(Convert.ToInt32(UserNumeric.Value), 192, DateTime.Now)
        End If
    End Sub

    'When the USER Textbox value changes, update the applicable OPC Tag
    Private Sub UserTextBox_TextChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles UserTextBox.TextChanged
        If (Not myOpcTags Is Nothing) Then
            myOpcTags("User.TextBox").SetVQT(UserTextBox.Text, 192, DateTime.Now)
        End If
    End Sub

    'The timer generates random values that are then sent to the SIMULATED controls
    Private Sub Timer1_Tick(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Timer1.Tick
        SimulationProgressBar.Value = Convert.ToInt32(randomizer.Next(100))
        SimulationNumeric.Value = Convert.ToDecimal(randomizer.Next(99999))
        SimulationCheckbox.Checked = Convert.ToBoolean(randomizer.Next(-1, 1))

        myOpcTags("Simulation.ProgressBar").SetVQT(Convert.ToInt32(SimulationProgressBar.Value), 192, DateTime.Now)
    End Sub

#End Region

#Region "   OPC SERVER EVENTS That WE Control "

    ' Event raised whenever an OPC Client connects
    Private Sub SlikServer1_OnClientConnect(ByVal sender As System.Object, ByVal eventArgs As NDI.SLIKDA.Interop.SLIKServer.OnClientConnectEventArgs) Handles SlikServer1.OnClientConnect
        ConnectedClients.Value = eventArgs.NumClients
    End Sub

    ' Event raised whenever an OPC Client DISCONNECTS
    Private Sub SlikServer1_OnClientDisconnect(ByVal sender As System.Object, ByVal eventArgs As NDI.SLIKDA.Interop.SLIKServer.OnClientDisconnectEventArgs) Handles SlikServer1.OnClientDisconnect
        ConnectedClients.Value = eventArgs.NumClients
    End Sub

    ' Event raised when an OPC client requests to WRITE some values to some Tags
    ' You are then responsible for 'making this happen', which could mean:
    '   (a) you write the values to some PLC/DCS system, i.e. some kind of hardware
    '   (b) you update the screen, as in this example
    '   (c) you update some other software etc.
    Private Sub SlikServer1_OnWrite( _
        ByVal sender As System.Object, _
        ByVal eventArgs As SLIKServer.OnWriteEventArgs) Handles SlikServer1.OnWrite



        '
        ' Iterate thru each/every item that the OPC Client has requested us to WRITE
        For i As Integer = 0 To (eventArgs.Count - 1)

            ' Get the "next" item in the list
            Dim currentItem As ISLIKTag = eventArgs.Tags(i)
            Dim currentValue As Object = eventArgs.Values(i)

            ' We will look at the name of the OPC Tag to then figure out WHERE
            ' to SEND the data to....
            Select Case (currentItem.Name)

                ' Checkbox on the LEFT side of the form
                Case "User.CheckBox"
                    UserCheckBox.Checked = Convert.ToBoolean(currentValue)


                    ' Numeric control on the LEFT side of the form
                Case "User.Numeric"
                    UserNumeric.Value = Convert.ToInt32(currentValue)


                    ' Textbox on the LEFT side of the form
                Case "User.TextBox"
                    UserTextBox.Text = Convert.ToString(currentValue)


                    ' Progress bar on the RIGHT side of the form
                Case "Simulation.ProgressBar"
                    SimulationProgressBar.Value = Convert.ToInt32(currentValue)


                    ' Numeric control on the RIGHT side of the form
                Case "Simulation.Numeric"
                    SimulationNumeric.Value = Convert.ToInt32(currentValue)


                    ' Checkbox on the RIGHT side of the form
                Case "Simulation.CheckBox"
                    SimulationCheckbox.Checked = Convert.ToBoolean(currentValue)
            End Select

            ' Specify that the Item at *this* position is NOT in error!
            eventArgs.Errors(i) = OPCDAErrorsEnum.sdaSOK
        Next

        ' Now specify that we completed this event successfully
        eventArgs.Result = OPCDAErrorsEnum.sdaSOK


    End Sub

    ' Event raised when an OPC client requests to READ the values to some Tags
    ' You are then responsible for 'making this happen', which could mean:
    '   (a) you READ the values from some PLC/DCS system, i.e. some kind of hardware
    '   (b) you READ the values fom the screen, as in this example
    '   (c) you READ the values from some other software etc.
    Private Sub SlikServer1_OnRead( _
        ByVal sender As System.Object, _
        ByVal eventArgs As SLIKServer.OnReadEventArgs) Handles SlikServer1.OnRead



        '
        ' Iterate thru each/every item that the OPC Client has requested us to READ
        For i As Integer = 0 To (eventArgs.Count - 1)

            ' Get the "next" item in the list
            Dim currentItem As ISLIKTag = eventArgs.Tags(i)

            ' Now to return the value of this tag, along with
            ' GOOD quality, and a current timestamp
            '   (Quality codes are: 192=GOOD; 0=BAD)
            '
            ' We will look at the name of the OPC Tag to then figure out WHERE
            ' to get the data from....
            Select Case (currentItem.Name)
                ' Checkbox on the LEFT side of the form
                Case "User.CheckBox"
                    eventArgs.Tags(i).SetVQT(UserCheckBox.Checked, 192, DateTime.Now)


                    ' Numeric control on the LEFT side of the form
                Case "User.Numeric"
                    eventArgs.Tags(i).SetVQT(Convert.ToInt32(UserNumeric.Value), 192, DateTime.Now)


                    ' Textbox on the LEFT side of the form
                Case "User.TextBox"
                    eventArgs.Tags(i).SetVQT(UserTextBox.Text, 192, DateTime.Now)


                    ' Progress bar on the RIGHT side of the form
                Case "Simulation.ProgressBar"
                    eventArgs.Tags(i).SetVQT(SimulationProgressBar.Value, 192, DateTime.Now)


                    ' Numeric control on the RIGHT side of the form
                Case "Simulation.Numeric"
                    eventArgs.Tags(i).SetVQT(Convert.ToInt32(SimulationNumeric.Value), 192, DateTime.Now)


                    ' Checkbox on the RIGHT side of the form
                Case "Simulation.CheckBox"
                    eventArgs.Tags(i).SetVQT(SimulationCheckbox.Checked, 192, DateTime.Now)
            End Select

            ' Specify that the Item at *this* position is NOT in error!
            eventArgs.Errors(i) = OPCDAErrorsEnum.sdaSOK
        Next

        ' Now specify that we completed this event successfully
        eventArgs.Result = OPCDAErrorsEnum.sdaSOK


    End Sub

    'MANY OTHER EVENTS DO EXIST, ADDING EVEN MORE POWER AND FUNCTIONALITY

#End Region

End Class